----------------------------------------------------------------------------------------------------------------------------------------

--   Tabelle für Audit/Tabellen-Log
    -- DROP TABLE IF EXISTS tlog.Auditlog;
CREATE TABLE tlog.Auditlog (
  l_id                     SERIAL PRIMARY KEY,
  l_txid                   BIGINT,                                       -- Transaktions-ID der Transaktion, die den Logeintrag ausgelöst hat.
  l_tablename              VARCHAR(100) NOT NULL,                        -- Schema + Tabellenname z.Bsp. "public.adk"
  l_user                   VARCHAR(30) DEFAULT tsystem.current_user_ll_db_usename(true),             -- DB-Benutzer der den Datensatz geändert hat
  l_time                   TIMESTAMP(0) NOT NULL DEFAULT currenttime(),  -- Zeitstempel der Änderung
  l_operation              VARCHAR(1),                                   -- 'I' oder 'U' oder 'D'  entsprechend TG_OP des auslösenden Triggers
  l_json_old               JSON,                                         -- nur geänderte Felder in JSON - Representation VOR der Änderung
  l_json_new               JSON,                                         -- nur geänderte Felder in JSON - Representation NACH der Änderung
  l_dbrid                  VARCHAR(32),                                  -- DBRID des geänderten Datensatzes
  l_parent_ID              VARCHAR,                                      -- Value von Parent-ID (z.B für op2 -> o2_ix, op6 -> o6_ix)
  l_parent_FName           VARCHAR,                                      -- Feldname
  l_references_tablename   VARCHAR(100),
  l_references_FName       VARCHAR,
  l_references_id          VARCHAR,
  l_ident                  VARCHAR,                                      -- Identifikator
  l_ident_json             json
  -- System (tables__generate_missing_fields)
  --   kein automatisches dbrid, insert_date, insert_by, modified_by, modified_date und table_delete-Trigger (tables__fieldInfo__fetch)
 );

 CREATE INDEX auditlog_l_parent_ID  ON tlog.auditlog(l_parent_ID);
 CREATE INDEX auditlog_l_tablename  ON tlog.auditlog(l_tablename);
 CREATE INDEX auditlog_l_dbrid      ON tlog.auditlog(l_dbrid);
 CREATE INDEX auditlog_l_txid       ON tlog.auditlog(l_txid);

--
CREATE OR REPLACE FUNCTION tlog.log_audit_enable() RETURNS VOID AS $$
  BEGIN
    IF current_setting('prodat.log_audit_count')::INTEGER > 0 THEN
        PERFORM set_config('prodat.log_audit_count', (current_setting('prodat.log_audit_count'):: INTEGER - 1)::VARCHAR, false);
    ELSEIF current_setting('prodat.log_audit_count')::INTEGER < 0 THEN
        PERFORM set_config('prodat.log_audit_count', '0', false);
    END IF;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION tlog.log_audit_disable() RETURNS VOID AS $$
  BEGIN
    PERFORM set_config('prodat.log_audit_count', (current_setting('prodat.log_audit_count')::INTEGER + 1)::VARCHAR, false);
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION tlog.log_audit_active() RETURNS BOOLEAN AS $$
  DECLARE curr_log_audit_count INTEGER;
  BEGIN
    BEGIN
        curr_log_audit_count:=current_setting('prodat.log_audit_count')::INTEGER;
    EXCEPTION WHEN OTHERS THEN
        PERFORM TSystem.Log_Init();
        curr_log_audit_count:= current_setting('prodat.log_audit_count')::INTEGER;
    END;

    IF curr_log_audit_count > 0 THEN
        RETURN false;
    ELSE
        RETURN true;
    END IF;
  END $$ LANGUAGE plpgsql;
--


CREATE OR REPLACE FUNCTION tlog.log__json__old_new__compare() RETURNS void
  AS $f$
  BEGIN

  EXECUTE $func$

  -- Vergleich von Datensatz-Stand  'old' und 'new'
    -- IN-Parameter: Datensatz-Stand 'old' und 'new' als JSON
    -- OUT-Parameter: 'old' und 'new' JSONB von geänderten Feldern
  CREATE OR REPLACE FUNCTION tlog.log__json__old_new__compare(IN json_oldData json, IN json_newData json, OUT json_old jsonb, OUT json_new jsonb) RETURNS RECORD AS $$
    DECLARE oldData    VARCHAR;
            rec        RECORD;
            str        VARCHAR;
    BEGIN
      json_old :=  '{}'::jsonb;
      json_new :=  '{}'::jsonb;

      FOR rec IN SELECT * FROM json_each(json_newData) LOOP -- alle Tabellenfelder durchgehen
          IF ( (COALESCE(json_oldData ->> rec.key, '')::VARCHAR <> COALESCE(json_newData ->> rec.key, '')::VARCHAR) )
              AND instr(rec.key::VARCHAR, '_rtf') = 0
              AND (rec.key::VARCHAR <> 'modified_date')
              AND (rec.key::VARCHAR <> 'modified_by') -- Abgleich, ob die Feld-Value geändert ist.
          THEN
              json_old:= json_object_agg(rec.key, json_oldData ->> rec.key)::jsonb || json_old::jsonb;
              json_new:= json_object_agg(rec.key, json_newData ->> rec.key)::jsonb || json_new::jsonb;
          END IF;
      END LOOP;

      -- gibt leere json-Objekte statt NULL zurück
      json_old:= COALESCE(json_old, '{}'::jsonb);
      json_new:= COALESCE(json_new, '{}'::jsonb);

      RETURN;
    END $$ LANGUAGE plpgsql;
    ---
  $func$ --  EXECUTE end
  RETURN;
END $f$ LANGUAGE plpgsql;

SELECT tlog.log__json__old_new__compare();
---

--- #10936, #17591
CREATE OR REPLACE FUNCTION tlog.log_json__transl(
    IN _l_id integer,
    IN _old boolean DEFAULT false
  ) RETURNS character varying AS $$
  DECLARE
   result     VARCHAR := '';
   rec        RECORD;
  BEGIN
    FOR rec IN
        ( SELECT * FROM (
                SELECT
                  l_operation,
                  GetFieldAlias( jkv.key )::VARCHAR(40) AS Bezeichnung,
                  jkv.value_new  AS json_value
                FROM tlog.Auditlog
                  LEFT JOIN LATERAL tlog.json_each__key__value__setof_record( l_id ) AS jkv ON true
                WHERE l_id = _l_id
                ) AS sub
     )
    LOOP
      result := result || concat_ws(' = ', rec.Bezeichnung, rec.json_value) || '; ';
    END LOOP;

    RETURN result;
  END $$ LANGUAGE plpgsql;
--
--- #15151
--- SQL-Script aus l_json_old zu Datensatz wiederherstellen
--- Funktion erstellt SQL-Script (INSERT INTO ... VALUES ...) von gelöschten Datensatz
CREATE OR REPLACE FUNCTION tlog.Auditlog__json_old__into_sql(
    IN l_tablename    varchar,
    IN l_json_old     json
  ) RETURNS text AS $$

  DECLARE
      _rec         record;
      _sql_insert  text;
      _sql_values  text;
      --- JSON-Elemente gesamte Anzahl
      _count       integer;
      --- laufender Datensatz
      _i           integer := 0;
  BEGIN

    --- JSON-Elemente gesamte Anzahl
    _count := count(*)
      FROM (
        SELECT key FROM (
            SELECT * FROM json_each_text( l_json_old )
        ) AS sub
        WHERE key NOT IN ( 'dbrid', 'insert_date', 'insert_by', 'modified_date', 'modified_by' )
    ) AS sub1;

    _sql_insert := 'INSERT INTO ' || replace( l_tablename, 'public.', '' ) || '( ' || chr(10);
    _sql_values := 'VALUES ( ' || chr(10);

    --- JSON als Table: feldname, value_old
    FOR _rec IN
        SELECT key, value
        FROM (
          SELECT *
          FROM json_each_text( l_json_old )
        ) AS sub
        WHERE key NOT IN ( 'dbrid', 'insert_date', 'insert_by', 'modified_date', 'modified_by' )
    LOOP
      _i := _i + 1;
      --- rec.key -> Feldname
      _sql_insert  :=
          _sql_insert
          || '    '
          || _rec.key
          --- beim letzte Datensatz, Koma soll nicht erstellt werden
          || CASE WHEN _i < _count THEN ',' ELSE ' ' END
          || chr(10);

      --- rec.value -> Wert
      _sql_values  :=
          _sql_values
          || '    '
          || coalesce( quote_literal( _rec.value ), 'null' )
          --- beim letzte Datensatz, Koma soll nicht erstellt werden
          || CASE WHEN _i < _count THEN ',' ELSE ' ' END
          --- Leerzeichenkette
          || repeat( ' ', CASE
                            WHEN length( coalesce( _rec.value || '  ', 'null' ) ) > 20 THEN
                              5
                          ELSE
                              20 - length( coalesce( _rec.value || '  ', 'null' ) )
                          END
                    )
          || '-- '
          || coalesce( _rec.key, 'null' )
          || chr(10);

    END LOOP;

    RETURN _sql_insert || ')' || chr(10) || _sql_values || ')' || chr(10) || ';';
  END $$ LANGUAGE plpgsql STABLE;
---
--- #17591
--- Funktion liefert key, value von Auditlogdatensatz
CREATE OR REPLACE FUNCTION tlog.json_each__key__value__setof_record(
    IN _l_id integer,
    OUT key character varying,
    OUT value_new character varying,
    OUT value_old character varying
    ) RETURNS SETOF RECORD AS $$
  DECLARE rec            record;
          _json          json;
          _json_new      json;
          _json_old      json;
          _operation     varchar;
  BEGIN

    SELECT l_operation, l_json_new, l_json_old INTO _operation, _json_new, _json_old FROM tlog.auditlog WHERE l_id = _l_id;

    IF _operation = 'D' THEN
        _json := _json_old;
    ELSE
        _json := _json_new;
    END IF;

    FOR rec IN (select * from json_each( _json ) ) LOOP
        key       := rec.key;
        value_new := ( SELECT l_json_new -> key FROM tlog.auditlog WHERE l_id = _l_id );
        value_old := ( SELECT l_json_old -> key FROM tlog.auditlog WHERE l_id = _l_id );
        value_new := replace( value_new, '"', '' );
        value_old := replace( value_old, '"', '' );
        RETURN NEXT;
    END LOOP;

 END $$ LANGUAGE plpgsql STABLE;

--------- Log-Tabelle für Änderungen    ******* Ende *******
--- #17190    ASK kopieren
--- Funktion konvertiert JSON-Value in einem Text mit Koma-Trennzeichen. Derzeit ist unbenutzt (GEO: 31.01.2022)
CREATE OR REPLACE FUNCTION tlog.json_each__value__rec( IN _json json ) RETURNS text AS $$
  DECLARE rec     record;
          result  text := '';
  BEGIN

    FOR rec IN (select * from json_each( _json ) ) LOOP
        IF rec.key NOT IN ( 'dbrid', 'insert_date', 'insert_by', 'modified_date', 'modified_by' ) THEN
            IF result = '' THEN
                result := rec.value::text;
            ELSE
                result := result || ', ' || rec.value::text; --- || ' - ' || rec.key || '-';
            END IF;
        END IF;

    END LOOP;

    RETURN result;
 END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tlog.json__value__replace(
      IN _json_in         json,
      IN _fieldname       varchar,                --- Feldname, wohin neuer Wert ( _value_new ) überschreiben soll
      IN _value_new       varchar,
      IN _datatype_new    varchar = 'varchar'
  ) RETURNS jsonb AS $$
 DECLARE   _json_out  jsonb := '{}';
           rec        record;

 BEGIN

    FOR rec IN SELECT * FROM json_each(_json_in) LOOP -- alle Tabellenfelder durchgehen
--RAISE NOTICE 'rec.key = %, rec.value = %', rec.key, rec.value;
        IF rec.key = _fieldname THEN
            IF _datatype_new = 'integer' THEN
                _json_out := json_object_agg( rec.key, _value_new::integer )::jsonb || _json_out::jsonb;
            ELSIF _datatype_new = 'boolean' THEN
                _json_out := json_object_agg( rec.key, _value_new::boolean )::jsonb || _json_out::jsonb;
            ELSIF _datatype_new = 'null' THEN
                _json_out := json_object_agg( rec.key, null                )::jsonb || _json_out::jsonb;
            ELSIF _datatype_new = 'date' THEN
                _json_out := json_object_agg( rec.key, _value_new::date    )::jsonb || _json_out::jsonb;
            ELSE
                _json_out := json_object_agg( rec.key, _value_new          )::jsonb || _json_out::jsonb;
            END IF;
        ELSE
            _json_out := json_object_agg( rec.key, _json_in ->> rec.key )::jsonb || _json_out::jsonb;
        END IF;
    END LOOP;

    RETURN _json_out;

  END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION tlog.json__value__dbrid_insert_modified__replace( IN _json json ) RETURNS json AS $$
  BEGIN

    SELECT tlog.json__value__replace( _json, 'dbrid'        , nextval('db_id_seq'::regclass)::varchar          , 'varchar' ) INTO _json;
    SELECT tlog.json__value__replace( _json, 'insert_date'  , current_date::varchar                            , 'date'    ) INTO _json;
    SELECT tlog.json__value__replace( _json, 'insert_by'    , current_user::varchar                            , 'varchar' ) INTO _json;
    --- SELECT tlog.json__value__replace( _json, 'modified_date', to_char(current_timestamp, 'DD.MM.YYYY HH:MM:SS'), 'date'    ) INTO _json;
    SELECT tlog.json__value__replace( _json, 'modified_date', to_char(current_timestamp, 'YYYY-MM-DD HH:MM:SS'), 'date'    ) INTO _json;          --- Format ISO 8601
    SELECT tlog.json__value__replace( _json, 'modified_by'  , current_user::varchar                            , 'varchar' ) INTO _json;

    RETURN _json;
 END $$ LANGUAGE plpgsql;
---
CREATE OR REPLACE FUNCTION tlog.l_json__get_fieldvalue(
    _l_id      integer,
    _fieldname varchar,
    _jsin_old  boolean DEFAULT true
  )  RETURNS character varying AS $$

  DECLARE _json    json;
  BEGIN
      IF _jsin_old THEN
          SELECT l_json_old INTO _json FROM tlog.Auditlog WHERE l_id = _l_id;
      ELSE
          SELECT l_json_new INTO _json FROM tlog.Auditlog WHERE l_id = _l_id;
      END IF;

      RETURN ( SELECT value_
               FROM (
                     SELECT
                       TRIM( both '"' FROM ( json_each( _json ) ).value::VARCHAR ) AS value_,
                       ( json_each( _json ) ).key AS fname
                     FROM tlog.Auditlog
                     WHERE l_id = _l_id
                    ) AS sub
                 WHERE sub.fname = _fieldname LIMIT 1 );
  END $$ LANGUAGE plpgsql;